// NUMERICAL DESIGN LIMITED PROPRIETARY INFORMATION
//
// This software is supplied under the terms of a license agreement or
// nondisclosure agreement with Numerical Design Limited and may not 
// be copied or disclosed except in accordance with the terms of that 
// agreement.
//
//      Copyright (c) 1996-2004 Numerical Design Limited.
//      All Rights Reserved.
//
// Numerical Design Limited, Chapel Hill, North Carolina 27514
// http://www.ndl.com

// Per Pixel Lighting shader

// textures
texture NormalMap 
< 
    string NTM = "shader";
>;

texture BaseMap 
< 
    string NTM = "base";
>;

float4x4 WorldViewProjection    : WORLDVIEWPROJECTION;
float4x3 InvWorld               : INVWORLD;
float3 WSLightPos               : GLOBAL = { 0.0f, 0.0f, 100.0f };
float AmbientIntensity          : GLOBAL = 0.2f;

// Vertex Shaders
struct VS_OUTPUT
{
    float4 Position   : POSITION;
    float2 NormalTex  : TEXCOORD0;
    float2 BaseTex    : TEXCOORD1;
    float3 TSLightDir : TEXCOORD2;
};

VS_OUTPUT VS(float3 Position : POSITION,
    float3 Normal   : NORMAL,
    float3 Binormal : BINORMAL,
    float3 Tangent  : TANGENT,
    float2 TexCoord : TEXCOORD)
{
    VS_OUTPUT Out = (VS_OUTPUT)0;

    // Transform position
    Out.Position = mul(float4(Position, 1), WorldViewProjection);

    // Fill in texture coordinates
    Out.NormalTex = TexCoord;
    Out.BaseTex = TexCoord;
    
    // Calculate tangent-space light direction
    
    // Get model-space light position
    float3 MSLightPos = mul(float4(WSLightPos, 1), InvWorld);

    // Get model-space light direction
    float3 MSLightDir = MSLightPos - Position;

    // Transform into tangent-space light direction
    float fX = dot(Tangent, MSLightDir);
    float fY = dot(Binormal, MSLightDir);
    float fZ = dot(Normal, MSLightDir);
    Out.TSLightDir = normalize(float3(fX, fY, fZ)) * 0.5f + 0.5f;
     
    return Out;
}

struct VS_OUTPUT_NO_PS
{
    float4 Position   : POSITION;
    float3 TSLightDir : COLOR;
    float2 NormalTex  : TEXCOORD0;
    float2 BaseTex    : TEXCOORD1;
};

VS_OUTPUT_NO_PS VS_NO_PS(float3 Position : POSITION,
    float3 Normal   : NORMAL,
    float3 Binormal : BINORMAL,
    float3 Tangent  : TANGENT,
    float2 TexCoord : TEXCOORD)
{
    VS_OUTPUT_NO_PS Out = (VS_OUTPUT_NO_PS)0;

    // Perform calculations as if using PS
    VS_OUTPUT Temp = VS(Position, Normal, Binormal, Tangent, TexCoord);
    Out.Position = Temp.Position;
    Out.NormalTex = Temp.NormalTex;
    Out.BaseTex = Temp.BaseTex;
        
    // Normalize and bias tangent-space light direction into color
    Out.TSLightDir = Temp.TSLightDir * 0.5f + 0.5f; 
    
    return Out;
}

// Samplers
sampler BaseSampler = sampler_state
{ 
    Texture = (BaseMap);
    AddressU = Clamp;
    AddressV = Clamp;
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

sampler NormalSampler = sampler_state
{ 
    Texture = (NormalMap);
    AddressU = Clamp;
    AddressV = Clamp;
    MipFilter = LINEAR; 
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

// Pixel shader
float4 PS(VS_OUTPUT In) : COLOR
{
    float3 BiasedNormal = tex2D(NormalSampler, In.NormalTex);
    float3 Normal = BiasedNormal * 2.0f - 1.0f;
    
    float LightIntensity = saturate(dot(Normal, In.TSLightDir)) 
        + AmbientIntensity;
    
    float4 Base = tex2D(BaseSampler, In.BaseTex);

    return Base * LightIntensity;
}

// techniques
technique PerPixelLightingPS
<
    string NBTMethod = "ATI";
    bool UsesNiRenderState = true;
    bool UsesNiLightState = false;
>
{
    pass P0
    {
        VertexShader = compile vs_1_1 VS();
        PixelShader = compile ps_1_1 PS();

        AlphaBlendEnable    = FALSE;
    }
}

technique PerPixelLightingNoPS
<
    string NBTMethod = "ATI";
    bool UsesNiRenderState = true;
    bool UsesNiLightState = false;
>
{
    pass P0
    {
        VertexShader = compile vs_1_1 VS_NO_PS();
        PixelShader = 0;

        Texture[0] = (NormalMap);
        ColorOp[0] = DotProduct3;
        ColorArg1[0] = Texture;
        ColorArg2[0] = Diffuse;
        AlphaOp[0] = SelectArg1;
        AlphaArg1[0] = Texture;
        AlphaArg2[0] = Diffuse;
        AddressU[0] = Clamp;
        AddressV[0] = Clamp;
        MipFilter[0] = LINEAR;
        MinFilter[0] = LINEAR;
        MagFilter[0] = LINEAR;

        Texture[1] = (BaseMap);
        ColorOp[1] = Modulate;
        ColorArg1[1] = Texture;
        ColorArg2[1] = Current;
        AlphaOp[1] = SelectArg1;
        AlphaArg1[1] = Texture;
        AlphaArg2[1] = Current;
        AddressU[1] = Clamp;
        AddressV[1] = Clamp;
        MipFilter[1] = LINEAR;
        MinFilter[1] = LINEAR;
        MagFilter[1] = LINEAR;

        AlphaBlendEnable = FALSE;
    }
}
