# Sid Meier's Civilization 4 
# Copyright Firaxis Games 2005 
# 
# Additional information: http://session.apolyton.net/modiki/index.php?title=Main_Page

from __future__ import division
import os
import wx
import wx.lib.imagebrowser as ib
import Image
import random

version = 18.6
Debug = 1

# BMP to WBS Converter Version 18.6
# Authors - Jon Shafer, Gabriele "Rhye" Trovato, Jesse Smith, Wouter "Locutus" Snijders
# Utility designed to convert a group of .bmp images to Civilization IV World Builder save data
# To run this utility you must have PyPIL installed (http://www.python.org/pypi/PIL/1.1.5)


COLOR_DARK_BLUE		= (0,	0,	128)
COLOR_DARK_GREEN 	= (0, 	128,    0)
COLOR_OLIVE     	= (128, 128,    0)
COLOR_DARK_YELLOW 	= (192, 192,    0)
COLOR_LIGHT_YELLOW 	= (255,	255,    128)
COLOR_LIGHT_GREY	= (192,	192,    192)
COLOR_WHITE		= (255, 255,    255)
COLOR_BLUE		= (0,	0,      255)
COLOR_LIGHT_BLUE	= (128,	128,    255)
COLOR_RED		= (255,	0,      0)
COLOR_RUST		= (128,	0,      0)
COLOR_YELLOW		= (255,	255,    0)
COLOR_DARK_GREEN_GREY	= (0,	64,     64)
COLOR_LIGHT_BLUE_GREY	= (0,	128,    128)
COLOR_LIGHT_RED		= (255,	128,    128)
COLOR_DARK_PINK 	= (128,	64,     64)
COLOR_PALE_MAGENTA   	= (255,	128,    255)
COLOR_LIGHT_MAGENTA    	= (255,	64,     255)
COLOR_MAGENTA    	= (255,	0,     255)
COLOR_DARK_PURPLE       = (128,	0,      128)
COLOR_PURPLE            = (192,	0,      192)
COLOR_CRIMSON    	= (192,	0,      64)
COLOR_DARK_BROWN 	= (64,	0,      0)
COLOR_BROWN             = (128,	64,     0)
COLOR_LIGHT_BROWN       = (192,	128,    64)
COLOR_PALE_BRICK_RED    = (192,	128,    128)
COLOR_CREAM             = (255,	255,    192)
COLOR_AMBER     	= (255,	192,    0)
COLOR_ORANGE     	= (255,	128,    0)
COLOR_DARK_ORANGE       = (255,	64,     0)
COLOR_DARK_RED          = (192,	0,      0)
COLOR_DARK_TAN          = (128,	128,    64)
COLOR_TAN               = (192,	192,    128)
COLOR_DARK_OLIVE        = (32,	64,     0)
COLOR_LIME              = (192,	255,    0)
COLOR_GREEN             = (0,	192,    0)
COLOR_LIGHT_GREEN       = (0,	255,    0)
COLOR_TEA_GREEN         = (222,	255,    192)
COLOR_PALE_TURQUOISE    = (64,	255,    192)
COLOR_DARK_CYAN       	= (0,	192,    192)
COLOR_CYAN       	= (0,	255,    255)
COLOR_PALE_CYAN         = (192,	255,    255)
COLOR_PALE_BLUE         = (128,	192,    255)
COLOR_DARK_NAVY_BLUE    = (0,	0,      64)
COLOR_BLUEISH_GREY      = (128,	128,    192)
COLOR_PALE_GREY	        = (222,	222,    222)
COLOR_GREY	        = (128,	128,    128)
COLOR_DARK_GREY	        = (64,	64,     64)
COLOR_BLACK      	= (0,	0,      0)

d_TerrainColorPalettes = {
	'TERRAIN_GRASS' :	COLOR_DARK_GREEN,
	'TERRAIN_PLAINS' : 	COLOR_OLIVE,
	'TERRAIN_DESERT' :	COLOR_LIGHT_YELLOW,
	'TERRAIN_TUNDRA' : 	COLOR_LIGHT_GREY,
	'TERRAIN_SNOW' : 	COLOR_WHITE,
	}

d_HeightColorPalettes = {
	'TERRAIN_PEAK' : COLOR_RUST,
	'TERRAIN_HILL' : COLOR_CRIMSON,
	'TERRAIN_GRASS': COLOR_DARK_GREEN,
	'TERRAIN_OCEAN': COLOR_DARK_BLUE,
	'TERRAIN_COAST': COLOR_BLUE,
	}

d_FeatureColorPalettes = {
	'FEATURE_JUNGLE' : COLOR_DARK_GREEN_GREY,
	'FEATURE_FOREST' : COLOR_LIGHT_BLUE_GREY,
	'FEATURE_ICE' : COLOR_LIGHT_BLUE,
	'FEATURE_OASIS' : COLOR_PALE_CYAN,
	'FEATURE_FLOOD_PLAINS' : COLOR_LIME,
	'FEATURE_FALLOUT' : COLOR_DARK_OLIVE,
	}	

d_RiverColorPalettes = {
	'RIVER_N_TO_S' : COLOR_RED,
	'RIVER_S_TO_N' : COLOR_YELLOW,
	'RIVER_E_TO_W' : COLOR_WHITE,
	'RIVER_W_TO_E' : COLOR_DARK_GREEN_GREY,
	'RIVER_ANGLE_W_TO_N' : COLOR_OLIVE,
	'RIVER_ANGLE_N_TO_W' : COLOR_LIGHT_RED,
	'RIVER_ANGLE_S_TO_NW' : COLOR_LIGHT_YELLOW,
	'RIVER_ANGLE_NW_TO_S' : COLOR_DARK_PINK,    
	}

d_BonusColorPalettes = {
	'BONUS_ALUMINUM' : COLOR_GREY,
	'BONUS_COAL' : COLOR_BLACK,
	'BONUS_COPPER' : COLOR_PALE_MAGENTA,
	'BONUS_HORSE' : COLOR_LIGHT_BROWN,
	'BONUS_IRON' : COLOR_BLUEISH_GREY,
	'BONUS_MARBLE' : COLOR_CYAN,
	'BONUS_OIL' : COLOR_DARK_NAVY_BLUE,
	'BONUS_STONE' : COLOR_TAN,
	'BONUS_URANIUM' : COLOR_LIGHT_GREEN,
	'BONUS_DYE' : COLOR_LIGHT_MAGENTA,
	'BONUS_FUR' : COLOR_DARK_BROWN,
	'BONUS_GEMS' : COLOR_PALE_TURQUOISE,
	'BONUS_GOLD' : COLOR_DARK_YELLOW,
	'BONUS_INCENSE' : COLOR_PALE_BRICK_RED,
	'BONUS_IVORY' : COLOR_DARK_TAN,
	'BONUS_SILK' : COLOR_PALE_BLUE,
	'BONUS_SILVER' : COLOR_PALE_GREY,
	'BONUS_SPICES' : COLOR_DARK_ORANGE,
	'BONUS_SUGAR' : COLOR_GREEN,
	'BONUS_WINE' : COLOR_PURPLE,
	'BONUS_WHALE' : COLOR_DARK_PURPLE,      
	'BONUS_BANANA' : COLOR_YELLOW,
	'BONUS_CLAM' : COLOR_DARK_GREEN_GREY,
	'BONUS_CORN' : COLOR_ORANGE,
	'BONUS_COW' : COLOR_DARK_GREY,
	'BONUS_CRAB' : COLOR_DARK_RED,
	'BONUS_DEER' : COLOR_BROWN,
	'BONUS_FISH' : COLOR_DARK_CYAN,
	'BONUS_PIG' : COLOR_LIGHT_RED,
	'BONUS_RICE' : COLOR_TEA_GREEN,
	'BONUS_SHEEP' : COLOR_CREAM,
	'BONUS_WHEAT' : COLOR_AMBER,
	}

d_ColorPalettes = {
	'FOW_LAND' : COLOR_PALE_MAGENTA,
	'FOW_WATER' : COLOR_MAGENTA,
	}



scenario = None

def runBMPConvertToWBSApp(scenarioObj):
	global scenario
	scenario = scenarioObj
	
	app = BMPConvertToWBSApp(0)
	app.MainLoop()

class BMPConvertToWBSFrame(wx.Frame):
	ID_SETUPMAP = 101
	ID_LOAD = 102
	ID_START = 103
	ID_FOW = 104
	ID_EXIT = 105
	ID_SETDEBUG = 106
	
	def __init__(self, parent, ID, title):
		self.wxApp = parent
		wx.Frame.__init__(self, parent, ID, title, (300,100), size=(600,650))
		
		self.hSizer	= wx.BoxSizer( wx.VERTICAL )
		self.SetSizer( self.hSizer )

		menuBar = wx.MenuBar()
		menu = wx.Menu()
		menu.Append(self.ID_SETUPMAP, "&Set BMP Converter Options", "Setup the map conversion variables")
		menu.Append(self.ID_LOAD, "&Load BMP Converter Options", "Loads the map conversion variables from the Scenario file")
		menu.Append(self.ID_START, "&Convert to WBS", "Converts the Bitmaps to a WBS file")
		menu.Append(self.ID_FOW, "&Extract Fog of War", "Extract the coordinates of the pink (revealed) pixels from a bitmap and to print them in a separate file")
		menu.Append(self.ID_EXIT, "&Exit", "Terminates the program")
		
		self.Bind(wx.EVT_MENU, self.OnSetupMap, id=self.ID_SETUPMAP)
		self.Bind(wx.EVT_MENU, self.OnLoad, id=self.ID_LOAD)
		self.Bind(wx.EVT_MENU, self.OnStart, id=self.ID_START)
		self.Bind(wx.EVT_MENU, self.OnFow, id=self.ID_FOW)
		self.Bind(wx.EVT_MENU, self.OnExit, id=self.ID_EXIT)
		
		menuBar.Append(menu, "&File")
		self.SetMenuBar(menuBar)
		
		self.CreateStatusBar()
		self.SetStatusText("Convert Bitmaps to a WBS file")
	
	def setApp(self, application):
		self.wxApp = application
	
	def OnSetupMap(self, event):
		global scenario
		self.setMapSize(self.wxApp.setXSize, "Step 1: Width", "Set X (Width) Size", scenario.iDefaultWidth)
		self.setMapSize(self.wxApp.setYSize, "Step 2: Height", "Set Y (Height) Size", scenario.iDefaultHeight)
		self.setMapSize(self.wxApp.setRiverSensitivity, "Step 3: River Sensitivity", "Set River Sensitivity 1 to 7:\nHigher means FEWER river plots are put on the map based upon the RiverMap", scenario.iDefaultRiverSensitivity)
		self.setMapSize(self.wxApp.setCoastsFlag, "Step 4: Coasts", "Set Coasts Assignment:\n0: Disabled (no coasts); 1: Manual only; 2: Automatic only; 3: Manual + Automatic", scenario.iDefaultCoastsAssignment)
		self.setMapSize(self.wxApp.setFloodPlainsFlag,"Step 5: Flood Plains", "Set Flood Plains Assignment:\n0: Disabled (no flood plains); 1: Manual only; 2: Automatic only; 3: Manual + Automatic", scenario.iDefaultFloodPlainsAssignment)
		self.setMapSize(self.wxApp.setWrapX, "Step 6: Wrap X", "Allow X-Wrapping: 1 = Yes; 0 = No", scenario.iDefaultXWrapping)
		self.setMapSize(self.wxApp.setWrapY, "Step 7: Wrap Y", "Allow Y-Wrapping: 1 = Yes; 0 = No", scenario.iDefaultYWrapping)
		self.setMapSize(self.wxApp.setNumberOfCivs, "Step 8: Number of players", "Set number of civs: 1 to 18", scenario.iDefaultNumberOfCivs)
		self.getBitMapPath(self.wxApp.setHeightMap, "Step 9: Height Map", "Select Height Map", True)
		self.getBitMapPath(self.wxApp.setTerrainMap, "Step 10: Terrain Map", "Select Terrain Map")
		self.getBitMapPath(self.wxApp.setFeatureMap, "Step 11: Feature Map", "Select Feature Map")
		self.getBitMapPath(self.wxApp.setRiverMap, "Step 12: River Map", "Select River Map")
		self.getBitMapPath(self.wxApp.setBonusMap, "Step 13: Bonus Map", "Select Bonus Map")

	def OnFow(self, event):
                self.wxApp.setFOWMap("") 
                self.getBitMapPath(self.wxApp.setFOWMap, "Select Map", "Select Map", True) 
                err = self.wxApp.FOWExtract() 
                if not err:  
                        print("Done! Open FOW_Output.txt") 


	def OnLoad(self, event):
		global scenario
                self.wxApp.setXSize( scenario.iDefaultWidth )
                self.wxApp.setYSize( scenario.iDefaultHeight )
		self.wxApp.setRiverSensitivity( scenario.iDefaultRiverSensitivity )
		self.wxApp.setCoastsFlag( scenario.iDefaultCoastsAssignment )
		self.wxApp.setFloodPlainsFlag( scenario.iDefaultFloodPlainsAssignment )
		self.wxApp.setWrapX( scenario.iDefaultXWrapping )
                self.wxApp.setWrapY( scenario.iDefaultYWrapping )
                self.wxApp.setNumberOfCivs( scenario.iDefaultNumberOfCivs )
		self.wxApp.setHeightMap( scenario.HeightMapPath )
		self.wxApp.setTerrainMap( scenario.TerrainMapPath )
		self.wxApp.setFeatureMap( scenario.FeatureMapPath )
		self.wxApp.setRiverMap( scenario.RiverMapPath )
		self.wxApp.setBonusMap( scenario.BonusMapPath )

	
	def setMapSize(self, AppCall, title, message, iDefaultValue):
		dlg = wx.TextEntryDialog(self, message, title, 'Python')
		dlg.SetValue(str(iDefaultValue))
		if dlg.ShowModal() == wx.ID_OK:
			AppCall( int(dlg.GetValue()) )
		
	def getBitMapPath(self, AppCall, title, message, bOK = False):
		# Inform the player what file is needed
		dlgFlag = wx.YES_NO
		
		if bOK:
			dlgFlag = wx.OK
		dlg = wx.MessageDialog(self, message, title, dlgFlag | wx.ICON_INFORMATION)
		if dlg.ShowModal() == wx.ID_NO:
			return
		dlg.Destroy()
		
		# display image browser
		dir = os.getcwd()
		dlg = ib.ImageDialog(self, dir)
		dlg.Centre()
		if dlg.ShowModal() == wx.ID_OK:
			AppCall( dlg.GetFile() )
	
	def setWBSFilePath(self, AppCall, message):
		dlg = 0
		dlg = wx.FileDialog( self,	message, wildcard =	"WBS Files (*.Civ4WorldBuilderSave)|*.Civ4WorldBuilderSave", style = wx.SAVE | wx.OVERWRITE_PROMPT )
		if dlg.ShowModal()	== wx.ID_OK:
			self.wxApp.setWBSPath( dlg.GetPath()	)
		dlg.Destroy()
#		self.Close(True)

	def OnStart(self, event):
		self.setWBSFilePath(self.wxApp.setWBSPath, "Set WBS Output File")
		print("\n")
		err = self.wxApp.ConvertBMP()
		if not err:
			print("Done!")

	def OnExit(self, event):
		self.Close(True)

class BMPConvertToWBSApp(wx.App):
	iSizeX = 20
	iSizeY = 30
	riverSensitivity = 5   # 1 to 7, higher means FEWER river plots are put on the map based upon the RiverMap - 5 is recommended
        iFeatureCoastsFlag = 3
        iFeatureFloodPlainsFlag = 3
        iWrapX=1
        iWrapY=0
        iNumberOfCivs = 2
        
	HeightMap = ""
	TerrainMap = ""
	FeatureMap = ""
	RiverMap = ""
	BonusMap = ""
	WBSFile = ""
	DebugLog = ""

	iNumHeightColorPalettes = len(d_HeightColorPalettes)
	iNumTerrainColorPalettes = len(d_TerrainColorPalettes)
	iNumFeatureColorPalettes = len(d_FeatureColorPalettes)
	iNumRiverColorPalettes = len(d_RiverColorPalettes)
	iNumBonusColorPalettes = len(d_BonusColorPalettes)

	def OnInit(self):
		frame = BMPConvertToWBSFrame(None, -1, "BMP to WBS Converter v%d" % version)
		frame.setApp(self)
		frame.Show(True)
		self.SetTopWindow(frame)
		return True
	
	def setXSize(self, iXSize):
		print "X set to %s" % iXSize
		self.iSizeX = iXSize
	
	def setYSize(self, iYSize):
		print "Y set to %s" % iYSize
		self.iSizeY = iYSize
	
	def setRiverSensitivity(self, iRiverSensitivity):
		print "River Sensitivity set to %s" % iRiverSensitivity
		self.riverSensitivity = iRiverSensitivity

	def setCoastsFlag(self, iFeatureCoastsFlag):
		print "Coasts flag set to %s" % iFeatureCoastsFlag
		self.iFeatureCoastsFlag = iFeatureCoastsFlag

	def setFloodPlainsFlag(self, iFeatureFloodPlainsFlag):
		print "Flood Plains flag set to %s" % iFeatureFloodPlainsFlag
		self.iFeatureFloodPlainsFlag = iFeatureFloodPlainsFlag

	def setWrapX(self, iWrapX):
		print "Wrap X = %s" % iWrapX
		self.iWrapX = iWrapX

	def setWrapY(self, iWrapY):
		print "Wrap Y = %s" % iWrapY
		self.iWrapY = iWrapY
		
	def setNumberOfCivs(self, iNumberOfCivs):
		print "Number of Civs = %s" % iNumberOfCivs
		self.iNumberOfCivs = iNumberOfCivs
		
	def setHeightMap(self, filePath):
		print "Setting HeightMap to %s" % filePath
		self.HeightMap = filePath
	
	def setTerrainMap(self, filePath):
		print "Setting TerrainMap to %s" % filePath
		self.TerrainMap = filePath
	
	def setFeatureMap(self, filePath):
		print "Setting FeatureMap to %s" % filePath
		self.FeatureMap = filePath

	def setRiverMap(self, filePath):
		print "Setting RiverMap to %s" % filePath
		self.RiverMap = filePath

	def setBonusMap(self, filePath):
		print "Setting BonusMap to %s" % filePath
		self.BonusMap = filePath
		
	def setWBSPath(self, filePath):
		print "Setting WBSPath to %s" % filePath
		self.WBSFile = filePath

	def setFOWMap(self, filePath):
		#print "Setting FOW Map to %s" % filePath
		self.FOWMap = filePath
		

	def ConvertBMP(self):
		# Set up images to be loaded
		# height map must be loaded
		self.HeightMap = Image.open(self.HeightMap)
		if self.TerrainMap:
			self.TerrainMap = Image.open(self.TerrainMap)
		if self.FeatureMap:
			self.FeatureMap = Image.open(self.FeatureMap)
		if self.RiverMap:
			self.RiverMap = Image.open(self.RiverMap)
		if self.BonusMap:
			self.BonusMap = Image.open(self.BonusMap)
				
		self.BMPSize = self.HeightMap.size
		
		if Debug:
			self.DebugLog = open("DebugLog.txt", 'w')
			self.DebugLogging("\n__init__")
		
		# Set up output file
		self.OutputFile = open(self.WBSFile, 'w')
		# automatically makes it a valid WBS file
		self.OutputFile.write(scenario.getExtraWBSText_GameSettings())
		self.OutputFile.write(scenario.getExtraWBSText_Diplomacy())
		self.OutputFile.write(scenario.getExtraWBSText_Civs())
		self.OutputFile.write(self.getExtraWBSText_WorldSize())
		self.OutputFile.write(self.getExtraWBSText_Wrap())
		self.OutputFile.write(scenario.getExtraWBSText_WorldSettings(self))
		self.OutputFile.write("\n\n### Plot Info ###\n")

		# self.tDoneSize is the size of the map in CIV
		self.tDoneSize = (self.iSizeX,self.iSizeY)
		# self.fFactor is the ratio between the BMP images and the final CIV map
		self.fFactor = (self.BMPSize[0] / self.tDoneSize[0], self.BMPSize[1] / self.tDoneSize[1])

		# Determine how many image pixels are getting squished into each plot
		self.iNumPixelsPerPlot = (int(self.fFactor[0] + 0.5), int(self.fFactor[1] + 0.5))
		self.iRiverSensitivity = self.riverSensitivity * ((int(self.fFactor[0] + 0.5) + int(self.fFactor[1] + 0.5)) / 2) / 5

		# How many pixels we're taking into account for this plot... e.g. 29, 30 and 31 if self.iNumPixelsPerPlot is 3
		self.iXOffset = 0
		self.iYOffset = 0
		
		self.iActivePixelX = 0
		self.iActivePixelY = 0
		
		self.tTerrainPixel = ()
		self.tHeightPixel = ()
		self.tFeaturePixel = ()
		self.tRiverPixel = ()
		self.tBonusPixel = ()		
		
		# Create list which keeps track of how common each terrain type is in nearby pixels to determine what the plot will be
		self.lTerrainCount = []
		self.iTerrainCountDefaultValue = 0
		for i in range( self.iNumTerrainColorPalettes ):
			self.lTerrainCount.append( self.iTerrainCountDefaultValue ) 
		
		# Create list which keeps track of how common each terrain type is in nearby pixels to determine what the plot will be
		self.lHeightCount = []
		self.iHeightCountDefaultValue = 0
		for i in range( self.iNumHeightColorPalettes ):
			self.lHeightCount.append( self.iHeightCountDefaultValue )
		
		# Create list which keeps track of how common each feature type is in nearby pixels to determine what the plot will be
		self.lFeatureCount = []
		self.iFeatureCountDefaultValue = 0
		for i in range( self.iNumFeatureColorPalettes+1 ):
			self.lFeatureCount.append( self.iFeatureCountDefaultValue )
			# Count river stuff
			self.iRiverNtoSCount = 0
			self.iRiverStoNCount = 0
			self.iRiverEtoWCount = 0
			self.iRiverWtoECount = 0
			self.iRiverAngleWtoNCount = 0
			self.iRiverAngleNtoWCount = 0
			self.iRiverAngleStoNWCount = 0
			self.iRiverAngleNWtoSCount = 0
				
		# Count bonus stuff
		self.lBonusCount = []
		self.lBonusStats = []
		self.iBonusCountDefaultValue = 0
		for i in range( self.iNumBonusColorPalettes ):
			self.lBonusCount.append( self.iBonusCountDefaultValue )
			self.lBonusStats.append( self.iBonusCountDefaultValue )
		
		# Plot Data
		self.iFeatureType = -1
		self.iFeatureVariety = -1
		self.sFeatureDesc = ""

		self.iTerrainType = -1
		self.sTerrainDesc = ""
		self.sRiverExists = ""
		self.sRiverDirection = ""
		self.sRiverExists2 = ""
		self.sRiverDirection2 = ""
		self.iPlotType = -1
		self.sBonusType = ""
		
		# Temp Area Plot Data
		asTerrain = []
		asFeatures = []
		aiFeaturesV = []
		asRiverExists = []
		asRiverDirection = []
		asRiverExists2 = []
		asRiverDirection2 = []
		aiPlots = []
		asBonusType = []
				
		# Generate temp lists with 0s to enter into the final lists
		for i2 in range(self.iSizeY):
			asTerrain.append("TERRAIN")
			asFeatures.append("FEATURE")
			aiFeaturesV.append(-1)
			asRiverExists.append("")
			asRiverDirection.append("")
			asRiverExists2.append("")
			asRiverDirection2.append("")
			aiPlots.append(-1)
			asBonusType.append("")

		# Actual Area Plot Data
		self.aasTerrainTypes = []
		self.aasFeatureTypes = []
		self.aaiFeatureVTypes = []
		self.aasRiverExists = []
		self.aasRiverDirection = []
		self.aasRiverExists2 = []
		self.aasRiverDirection2 = []
		self.aaiPlotTypes = []
		self.aasBonusType = []
	
		# Copy temp lists to final lists
		for i3 in range(self.iSizeX):
			self.aasTerrainTypes.append(asTerrain[:])
			self.aasFeatureTypes.append(asFeatures[:])
			self.aaiFeatureVTypes.append(aiFeaturesV[:])
			self.aasRiverExists.append(asRiverExists[:])
			self.aasRiverDirection.append(asRiverDirection[:])
			self.aasRiverExists2.append(asRiverExists2[:])
			self.aasRiverDirection2.append(asRiverDirection[:])
			self.aaiPlotTypes.append(aiPlots[:])
			self.aasBonusType.append(asBonusType[:])

		self.ConvertImage()
		self.OutputFile.close()
		if Debug:
			self.DebugLog.close()
				
	def ConvertImage(self):
		self.DebugLogging("\nConvertImage")
                print("PROCESSING IMAGES...")
		# Loop through x and y for each plot on the map
		for self.iPlotX in range(self.tDoneSize[0]):
			print("%d percent completed" % (100 * self.iPlotX / self.tDoneSize[0]))
			for self.iPlotY in range(self.tDoneSize[1]):
				
				# Variables relating to each plot's data
				self.ResetPlotData()
				self.DebugLogging("\nX: %d \t Y: %d" %(self.iPlotX, self.iPlotY))
				
				# Loop through corresponding nearby pixels x and y for this plot
				for self.iPixelX in range(self.iNumPixelsPerPlot[0]):
					for self.iPixelY in range(self.iNumPixelsPerPlot[1]):

						# Counts how many pixels we're using and determine's the active pixel for this iteration of PixelX and PixelY
						self.DeterminePixels()
						
						# Verify pixel is not out of bounds
						if ((self.iActivePixelX >= 0 and self.iActivePixelX < self.BMPSize[0]) and (self.iActivePixelY >= 0 and self.iActivePixelY < self.BMPSize[1])):
							
							# Adds up the pixels of each type
							self.CountHeightPixels()
							self.CountTerrainPixels()
							self.CountFeaturePixels()
							self.CountRiverPixels()
							self.CountBonusPixels()
							

		# ****************************** DETERMINE PLOT DATA *****************************
				# Variables relating to each plot's data
				# Determines the actual data for each plot
				# Note: Order of these items determines weight given in case equalities occur
				self.DetermineHeightData()
				self.DetermineTerrainData()
				self.DetermineFeatureData()
				self.DetermineRiverData()
				self.DetermineBonusData()
				# Record this plot in the uber array
				self.aasTerrainTypes[self.iPlotX][self.iPlotY] = self.sTerrainDesc
				self.aasFeatureTypes[self.iPlotX][self.iPlotY] = self.sFeatureDesc
				self.aaiFeatureVTypes[self.iPlotX][self.iPlotY] = self.iFeatureVariety
				self.aasRiverExists[self.iPlotX][self.iPlotY] = self.sRiverExists
				self.aasRiverDirection[self.iPlotX][self.iPlotY] = self.sRiverDirection
				self.aasRiverExists2[self.iPlotX][self.iPlotY] = self.sRiverExists2
				self.aasRiverDirection2[self.iPlotX][self.iPlotY] = self.sRiverDirection2
				self.aaiPlotTypes[self.iPlotX][self.iPlotY] = self.iPlotType
				self.aasBonusType[self.iPlotX][self.iPlotY] = self.sBonusType

		self.DebugLogging("\nFlags: %d, %d" %(self.iFeatureCoastsFlag, self.iFeatureFloodPlainsFlag))
		
                print("WRITING FILE...")
		# Loop through x and y for each plot on the map again (could do this before were it not for AddCoastlines() step)
		self.DebugLogging("\nSize: %d, %d" %(self.tDoneSize[0], self.tDoneSize[1]))
		for iPlotX in range(self.tDoneSize[0]):
                    	print("%d percent completed" % (100 * iPlotX / self.tDoneSize[0]))
			for iPlotY in range(self.tDoneSize[1]):
				
				# If this plot is Ocean and next to land turn it into Coast instead
				if (self.iFeatureCoastsFlag == 2) or (self.iFeatureCoastsFlag == 3):
                                        self.AddCoastlines(iPlotX, iPlotY)
								
				# If this plot is Desert and next to a river add a Flood Plain
				if (self.iFeatureFloodPlainsFlag == 2) or (self.iFeatureFloodPlainsFlag == 3):
                                        self.AddFloodPlains(iPlotX, iPlotY)
								
				# Outputs this plot to the file
				self.OutputToFile(iPlotX, iPlotY)

		# Show bonuses statistics
                print("\nBonus Statistics:\n")
		print("ALUMINUM: %d" %(self.lBonusStats[0]))
		print("COAL: %d" %(self.lBonusStats[1]))
		print("COPPER: %d" %(self.lBonusStats[2]))
		print("HORSE: %d" %(self.lBonusStats[3]))
		print("IRON: %d" %(self.lBonusStats[4]))
		print("MARBLE: %d" %(self.lBonusStats[5]))
		print("OIL: %d" %(self.lBonusStats[6]))
		print("STONE: %d" %(self.lBonusStats[7]))
		print("URANIUM: %d" %(self.lBonusStats[8]))
		print("DYE: %d" %(self.lBonusStats[9]))
		print("FUR: %d" %(self.lBonusStats[10]))
		print("GEMS: %d" %(self.lBonusStats[11]))
		print("GOLD: %d" %(self.lBonusStats[12]))
		print("INCENSE: %d" %(self.lBonusStats[13]))
		print("IVORY: %d" %(self.lBonusStats[14]))
		print("SILK: %d" %(self.lBonusStats[15]))
		print("SILVER: %d" %(self.lBonusStats[16]))
		print("SPICES: %d" %(self.lBonusStats[17]))
		print("SUGAR: %d" %(self.lBonusStats[18]))
		print("WINE: %d" %(self.lBonusStats[19]))
		print("WHALE: %d" %(self.lBonusStats[20]))
		print("BANANA: %d" %(self.lBonusStats[21]))
		print("CLAM: %d" %(self.lBonusStats[22]))
		print("CORN: %d" %(self.lBonusStats[23]))
		print("COW: %d" %(self.lBonusStats[24]))
		print("CRAB: %d" %(self.lBonusStats[25]))
		print("DEER: %d" %(self.lBonusStats[26]))
		print("FISH: %d" %(self.lBonusStats[27]))
		print("PIG: %d" %(self.lBonusStats[28]))
		print("RICE: %d" %(self.lBonusStats[29]))
		print("SHEEP: %d" %(self.lBonusStats[30]))
		print("WHEAT: %d" %(self.lBonusStats[31]))		


	def DeterminePixels(self):
		self.DebugLogging("\nDeterminePixels")
		# How many pixels we're taking into account for this plot... e.g. 29, 30 and 31 if self.iNumPixelsPerPlot is 3
		self.iXOffset = self.iNumPixelsPerPlot[0] - int(self.iNumPixelsPerPlot[0] / 2)
		self.iYOffset = self.iNumPixelsPerPlot[1] - int(self.iNumPixelsPerPlot[1] / 2)
		
		# Pixel            =  Plot        * PixelFactor      + PixelLoop    + Offset
		self.iActivePixelX = (self.iPlotX * self.fFactor[0]) + self.iPixelX + self.iXOffset -1
		self.iActivePixelY = self.BMPSize[1] - ((self.iPlotY * self.fFactor[1]) + self.iPixelY + self.iYOffset)

	def CountHeightPixels(self):
		if not self.HeightMap:
			return
		self.DebugLogging("\nCountHeightPixels")
		self.tHeightPixel = self.HeightMap.getpixel((self.iActivePixelX, self.iActivePixelY))
		
		self.DebugLogging("\nHeightPixelCount: %s - " %(self.tHeightPixel,))
		if ( self.tHeightPixel == d_HeightColorPalettes.get('TERRAIN_PEAK') ):
			self.DebugLogging("Add Height Pixel: Peak")
			self.lHeightCount[0] += 1
		if ( self.tHeightPixel == d_HeightColorPalettes.get('TERRAIN_HILL') ):
			self.DebugLogging("Add Height Pixel: Hill")
			self.lHeightCount[1] += 1
		if ( self.tHeightPixel == d_HeightColorPalettes.get('TERRAIN_GRASS') ):
			self.DebugLogging("Add Height Pixel: Grass")
			self.lHeightCount[2] += 1
		if ( self.tHeightPixel == d_HeightColorPalettes.get('TERRAIN_OCEAN') or self.tHeightPixel == (0,0,254) ):
			self.DebugLogging("Add Height Pixel: Ocean")
			self.lHeightCount[3] += 1
		if ( self.tHeightPixel == d_HeightColorPalettes.get('TERRAIN_COAST') ):
			self.DebugLogging("Add Height Pixel: Coast")
			self.lHeightCount[4] += 1
			
	def CountTerrainPixels(self):
		if not self.TerrainMap:
			return
		self.DebugLogging("\nCountTerrainPixels")
		self.tTerrainPixel = self.TerrainMap.getpixel((self.iActivePixelX, self.iActivePixelY))
				
		self.DebugLogging("\nTerrainPixelCount: %s - " %(self.tTerrainPixel,))
		if ( self.tTerrainPixel == d_TerrainColorPalettes.get('TERRAIN_GRASS') ):
			self.DebugLogging("Add Terrain Pixel: Grass")
			self.lTerrainCount[0] += 1
		if ( self.tTerrainPixel == d_TerrainColorPalettes.get('TERRAIN_PLAINS') ):
			self.DebugLogging("Add Terrain Pixel: Plains")
			self.lTerrainCount[1] += 1
		if ( self.tTerrainPixel == d_TerrainColorPalettes.get('TERRAIN_DESERT') ):
			self.DebugLogging("Add Terrain Pixel: Desert")
			self.lTerrainCount[2] += 1
		if ( self.tTerrainPixel == d_TerrainColorPalettes.get('TERRAIN_TUNDRA') ):
			self.DebugLogging("Add Terrain Pixel: Tundra")
			self.lTerrainCount[3] += 1
		if ( self.tTerrainPixel == d_TerrainColorPalettes.get('TERRAIN_SNOW') ):
			self.DebugLogging("Add Terrain Pixel: Snow")
			self.lTerrainCount[4] += 1

	def CountFeaturePixels(self):
		if not self.FeatureMap:
			return
		self.DebugLogging("\nCountFeaturePixels")
		self.tFeaturePixel = self.FeatureMap.getpixel((self.iActivePixelX, self.iActivePixelY))
				
		self.DebugLogging("\nFeaturePixelCount: %s - " %(self.tFeaturePixel,))
		if ( self.tFeaturePixel == d_TerrainColorPalettes.get('TERRAIN_GRASS') ):
			self.DebugLogging("Add Feature Pixel: None")
			self.lFeatureCount[0] += 1
		if ( self.tFeaturePixel == d_FeatureColorPalettes.get('FEATURE_JUNGLE') ):
			self.DebugLogging("Add Feature Pixel: Jungle")
			self.lFeatureCount[1] += 1
		if ( self.tFeaturePixel == d_FeatureColorPalettes.get('FEATURE_FOREST') ):
			self.DebugLogging("Add Feature Pixel: Forest")
			self.lFeatureCount[2] += 1
		if ( self.tFeaturePixel == d_FeatureColorPalettes.get('FEATURE_ICE') ):
			self.DebugLogging("Add Feature Pixel: Ice")
			self.lFeatureCount[3] += 1
		if ( self.tFeaturePixel == d_FeatureColorPalettes.get('FEATURE_OASIS') ):
			self.DebugLogging("Add Feature Pixel: Oasis")
			self.lFeatureCount[4] += 1
		if ( self.tFeaturePixel == d_FeatureColorPalettes.get('FEATURE_FLOOD_PLAINS') ):
			self.DebugLogging("Add Feature Pixel: Flood Plain")
			self.lFeatureCount[5] += 1
		if ( self.tFeaturePixel == d_FeatureColorPalettes.get('FEATURE_FALLOUT') ):
			self.DebugLogging("Add Feature Pixel: Fallout")
			self.lFeatureCount[6] += 1
		
	def CountRiverPixels(self):
		if not self.RiverMap:
			return
		self.DebugLogging("\nCountRiverPixels")
		self.tRiverPixel = self.RiverMap.getpixel((self.iActivePixelX, self.iActivePixelY))

		self.DebugLogging("\nRiverPixelCount: %s - " %(self.tRiverPixel,))
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_N_TO_S')):
			self.DebugLogging("Add River Pixel: N to S")
			self.iRiverNtoSCount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_S_TO_N')):
			self.DebugLogging("Add River Pixel: S to N")
			self.iRiverStoNCount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_E_TO_W')):
			self.DebugLogging("Add River Pixel: E to W")
			self.iRiverEtoWCount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_W_TO_E')):
			self.DebugLogging("Add River Pixel: W to E")
			self.iRiverWtoECount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_ANGLE_W_TO_N')):
			self.DebugLogging("Add River Angle Pixel: W to N")
			self.iRiverAngleWtoNCount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_ANGLE_N_TO_W')):
			self.DebugLogging("Add River Angle Pixel: N to W")
			self.iRiverAngleNtoWCount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_ANGLE_S_TO_NW')):
			self.DebugLogging("Add River Angle Pixel: S to NW")
			self.iRiverAngleStoNWCount += 1
		if (self.tRiverPixel == d_RiverColorPalettes.get('RIVER_ANGLE_NW_TO_S')):
			self.DebugLogging("Add River Angle Pixel: NW to S")
			self.iRiverAngleNWtoSCount += 1
				
						
	def CountBonusPixels(self):
		if not self.BonusMap:
			return
		self.DebugLogging("\nCountBonusPixels")
		self.tBonusPixel = self.BonusMap.getpixel((self.iActivePixelX, self.iActivePixelY))
				
		self.DebugLogging("\nBonusPixelCount: %s - " %(self.tBonusPixel,))
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_ALUMINUM')):
			self.DebugLogging("Add Bonus Pixel: ALUMINUM")
			self.lBonusCount[0] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_COAL')):
			self.DebugLogging("Add Bonus Pixel: COAL")
			self.lBonusCount[1] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_COPPER')):
			self.DebugLogging("Add Bonus Pixel: COPPER")
			self.lBonusCount[2] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_HORSE')):
			self.DebugLogging("Add Bonus Pixel: HORSE")
			self.lBonusCount[3] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_IRON')):
			self.DebugLogging("Add Bonus Pixel: IRON")
			self.lBonusCount[4] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_MARBLE')):
			self.DebugLogging("Add Bonus Pixel: MARBLE")
			self.lBonusCount[5] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_OIL')):
			self.DebugLogging("Add Bonus Pixel: OIL")
			self.lBonusCount[6] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_STONE')):
			self.DebugLogging("Add Bonus Pixel: STONE")
			self.lBonusCount[7] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_URANIUM')):
			self.DebugLogging("Add Bonus Pixel: URANIUM")
			self.lBonusCount[8] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_DYE')):
			self.DebugLogging("Add Bonus Pixel: DYE")
			self.lBonusCount[9] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_FUR')):
			self.DebugLogging("Add Bonus Pixel: FUR")
			self.lBonusCount[10] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_GEMS')):
			self.DebugLogging("Add Bonus Pixel: GEMS")
			self.lBonusCount[11] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_GOLD')):
			self.DebugLogging("Add Bonus Pixel: GOLD")
			self.lBonusCount[12] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_INCENSE')):
			self.DebugLogging("Add Bonus Pixel: INCENSE")
			self.lBonusCount[13] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_IVORY')):
			self.DebugLogging("Add Bonus Pixel: IVORY")
			self.lBonusCount[14] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_SILK')):
			self.DebugLogging("Add Bonus Pixel: SILK")
			self.lBonusCount[15] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_SILVER')):
			self.DebugLogging("Add Bonus Pixel: SILVER")
			self.lBonusCount[16] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_SPICES')):
			self.DebugLogging("Add Bonus Pixel: SPICES")
			self.lBonusCount[17] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_SUGAR')):
			self.DebugLogging("Add Bonus Pixel: SUGAR")
			self.lBonusCount[18] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_WINE')):
			self.DebugLogging("Add Bonus Pixel: WINE")
			self.lBonusCount[19] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_WHALE')):
			self.DebugLogging("Add Bonus Pixel: WHALE")
			self.lBonusCount[20] += 1     
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_BANANA')):
			self.DebugLogging("Add Bonus Pixel: BANANA")
			self.lBonusCount[21] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_CLAM')):
			self.DebugLogging("Add Bonus Pixel: CLAM")
			self.lBonusCount[22] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_CORN')):
			self.DebugLogging("Add Bonus Pixel: CORN")
			self.lBonusCount[23] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_COW')):
			self.DebugLogging("Add Bonus Pixel: COW")
			self.lBonusCount[24] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_CRAB')):
			self.DebugLogging("Add Bonus Pixel: CRAB")
			self.lBonusCount[25] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_DEER')):
			self.DebugLogging("Add Bonus Pixel: DEER")
			self.lBonusCount[26] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_FISH')):
			self.DebugLogging("Add Bonus Pixel: FISH")
			self.lBonusCount[27] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_PIG')):
			self.DebugLogging("Add Bonus Pixel: PIG")
			self.lBonusCount[28] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_RICE')):
			self.DebugLogging("Add Bonus Pixel: RICE")
			self.lBonusCount[29] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_SHEEP')):
			self.DebugLogging("Add Bonus Pixel: SHEEP")
			self.lBonusCount[30] += 1
		if (self.tBonusPixel == d_BonusColorPalettes.get('BONUS_WHEAT')):
			self.DebugLogging("Add Bonus Pixel: WHEAT")
			self.lBonusCount[31] += 1
		
	def ResetPlotData(self):
		self.DebugLogging("\nResetPlotData")
		for i in range( self.iNumTerrainColorPalettes ):
			self.lTerrainCount[i] = 0
		for i in range( self.iNumHeightColorPalettes ):
			self.lHeightCount[i] = 0
		for i in range( self.iNumFeatureColorPalettes+1 ):
			self.lFeatureCount[i] = 0
		for i in range( self.iNumBonusColorPalettes ):
				self.lBonusCount[i] = 0
			
		self.iFeatureVariety = -1
		self.sFeatureDesc = "NONE"
		
		self.sTerrainDesc = "NONE"
		
		self.iRiverNtoSCount = 0
		self.iRiverStoNCount = 0
		self.iRiverEtoWCount = 0
		self.iRiverWtoECount = 0
		self.iRiverAngleWtoNCount = 0
		self.iRiverAngleNtoWCount = 0
		self.iRiverAngleStoNWCount = 0
		self.iRiverAngleNWtoSCount = 0
		
		self.sRiverExists = ""
		self.sRiverDirection = ""
		self.sRiverExists2 = ""
		self.sRiverDirection2 = ""              
		
		self.iPlotType = -1

		self.sBonusType = ""

	def DetermineHeightData(self):
		if not self.HeightMap:
			return
		self.DebugLogging("\nDetermineHeightData")
		# peak
		if (self.lHeightCount[0] >= self.lHeightCount[1]) and \
			(self.lHeightCount[0] >= self.lHeightCount[2]) and \
			(self.lHeightCount[0] >= self.lHeightCount[3]) and \
			(self.lHeightCount[0] >= self.lHeightCount[4]):
				self.DebugLogging("\nPlot Height set TERRAIN_PEAK")
				self.iPlotType = 0
		# hill
		elif (self.lHeightCount[1] >= self.lHeightCount[0]) and \
			(self.lHeightCount[1] >= self.lHeightCount[2]) and \
			(self.lHeightCount[1] >= self.lHeightCount[3]) and \
			(self.lHeightCount[1] >= self.lHeightCount[4]):
				self.DebugLogging("\nPlot Height set TERRAIN_HILL")
				self.iPlotType = 1
		
		# flatland
		elif (self.lHeightCount[2] >= self.lHeightCount[1]) and \
			(self.lHeightCount[2] >= self.lHeightCount[0]) and \
			(self.lHeightCount[2] >= self.lHeightCount[3]) and \
			(self.lHeightCount[2] >= self.lHeightCount[4]):
				self.DebugLogging("\nPlot Height set TERRAIN_GRASS")
				self.iPlotType = 2
		
		# ocean
		elif (self.lHeightCount[3] >= self.lHeightCount[0]) and \
			(self.lHeightCount[3] >= self.lHeightCount[1]) and \
			(self.lHeightCount[3] >= self.lHeightCount[2]) and \
			(self.lHeightCount[3] >= self.lHeightCount[4]):
				self.DebugLogging("\nPlot Height set TERRAIN_OCEAN")
				self.iPlotType = 3
				self.sTerrainDesc = 'TERRAIN_OCEAN'

		# coast (if enabled)
		elif (self.lHeightCount[4] >= self.lHeightCount[0]) and \
			(self.lHeightCount[4] >= self.lHeightCount[1]) and \
			(self.lHeightCount[4] >= self.lHeightCount[2]) and \
			(self.lHeightCount[4] >= self.lHeightCount[3]):
                                if ((self.iFeatureCoastsFlag == 1) or (self.iFeatureCoastsFlag == 3)):
                                        self.DebugLogging("\nPlot Height set TERRAIN_COAST")
                                        self.iPlotType = 3
                                        self.sTerrainDesc = 'TERRAIN_COAST'        
                                else:
                                        self.DebugLogging("\nPlot Height set TERRAIN_OCEAN")
                                        self.iPlotType = 3
                                        self.sTerrainDesc = 'TERRAIN_OCEAN'
								
	def DetermineTerrainData(self):
		if not self.TerrainMap:
			return
		self.DebugLogging("\nDetermineTerrainData")
		# if the terrain description was already handled, skip this
		if (self.sTerrainDesc != "NONE"):
			return
		
		if (self.lTerrainCount[0] >= self.lTerrainCount[1]) and \
			(self.lTerrainCount[0] >= self.lTerrainCount[2]) and \
			(self.lTerrainCount[0] >= self.lTerrainCount[3]) and \
			(self.lTerrainCount[0] >= self.lTerrainCount[4]):
				self.DebugLogging("\nPlot Terrain set TERRAIN_GRASS")
				self.sTerrainDesc = 'TERRAIN_GRASS'
			
		elif (self.lTerrainCount[1] >= self.lTerrainCount[0]) and \
			(self.lTerrainCount[1] >= self.lTerrainCount[2]) and \
			(self.lTerrainCount[1] >= self.lTerrainCount[3]) and \
			(self.lTerrainCount[1] >= self.lTerrainCount[4]):
				self.DebugLogging("\nPlot Terrain set TERRAIN_PLAINS")
				self.sTerrainDesc = 'TERRAIN_PLAINS'
		
		elif (self.lTerrainCount[2] >= self.lTerrainCount[1]) and \
			(self.lTerrainCount[2] >= self.lTerrainCount[0]) and \
			(self.lTerrainCount[2] >= self.lTerrainCount[3]) and \
			(self.lTerrainCount[2] >= self.lTerrainCount[4]):
				self.DebugLogging("\nPlot Terrain set TERRAIN_DESERT")
				self.sTerrainDesc = 'TERRAIN_DESERT'

		elif (self.lTerrainCount[3] >= self.lTerrainCount[1]) and \
			(self.lTerrainCount[3] >= self.lTerrainCount[2]) and \
			(self.lTerrainCount[3] >= self.lTerrainCount[0]) and \
			(self.lTerrainCount[3] >= self.lTerrainCount[4]):
				self.DebugLogging("\nPlot Terrain set TERRAIN_TUNDRA")
				self.sTerrainDesc = 'TERRAIN_TUNDRA'
		
		elif (self.lTerrainCount[4] >= self.lTerrainCount[1]) and \
			(self.lTerrainCount[4] >= self.lTerrainCount[2]) and \
			(self.lTerrainCount[4] >= self.lTerrainCount[3]) and \
			(self.lTerrainCount[4] >= self.lTerrainCount[0]):
				self.DebugLogging("\nPlot Terrain set TERRAIN_SNOW")
				self.sTerrainDesc = 'TERRAIN_SNOW'
	
	def DetermineFeatureData(self):
		if not self.FeatureMap:
			return
		self.DebugLogging("\nDetermineFeatureData")
		self.DebugLogging("\n0: %d\nJ: %d\nF: %d\nI: %d\nO: %d\nFP: %d" %(self.lFeatureCount[0], self.lFeatureCount[1], self.lFeatureCount[2], self.lFeatureCount[3], self.lFeatureCount[4], self.lFeatureCount[5]))

		# No Feature
		if (self.lFeatureCount[0] >= self.lFeatureCount[1]) and \
			(self.lFeatureCount[0] >= self.lFeatureCount[2]) and \
			(self.lFeatureCount[0] >= self.lFeatureCount[3]) and \
			(self.lFeatureCount[0] >= self.lFeatureCount[4]) and \
			(self.lFeatureCount[0] >= self.lFeatureCount[5]) and \
			(self.lFeatureCount[0] >= self.lFeatureCount[6]):
				self.iFeatureVariety = -1
				self.sFeatureDesc = "NONE"
				return

		
		# Jungle and forest cannot appear in desert
		if (self.sTerrainDesc != 'TERRAIN_DESERT'):
			# Jungle and forest can only appear on flatland or hills
			if (self.iPlotType == 1) or (self.iPlotType == 2):
				if (self.lFeatureCount[1] >= self.lFeatureCount[0]) and \
					(self.lFeatureCount[1] >= self.lFeatureCount[2]) and \
					(self.lFeatureCount[1] >= self.lFeatureCount[3]) and \
					(self.lFeatureCount[1] >= self.lFeatureCount[4]) and \
					(self.lFeatureCount[1] >= self.lFeatureCount[5]) and \
					(self.lFeatureCount[1] >= self.lFeatureCount[6]):
						self.DebugLogging("\nPlot Feature set FEATURE_JUNGLE")
						self.iFeatureVariety = 0
						self.sFeatureDesc = 'FEATURE_JUNGLE'
						return
						
				elif (self.lFeatureCount[2] >= self.lFeatureCount[0]) and \
					(self.lFeatureCount[2] >= self.lFeatureCount[1]) and \
					(self.lFeatureCount[2] >= self.lFeatureCount[3]) and \
					(self.lFeatureCount[2] >= self.lFeatureCount[4]) and \
					(self.lFeatureCount[2] >= self.lFeatureCount[5]) and \
					(self.lFeatureCount[2] >= self.lFeatureCount[6]):
						self.DebugLogging("\nPlot Feature set FEATURE_FOREST")
						self.sFeatureDesc = 'FEATURE_FOREST'
						if (self.sTerrainDesc == 'TERRAIN_DESERT') or (self.sTerrainDesc == 'TERRAIN_PLAINS'):
							self.iFeatureVariety = 0
						elif (self.sTerrainDesc == 'TERRAIN_GRASS'):
                                                        self.iFeatureVariety = random.randint(0, 1)                                        
						elif (self.sTerrainDesc == 'TERRAIN_TUNDRA'):
							self.iFeatureVariety = random.randint(1, 2) 
						elif (self.sTerrainDesc == 'TERRAIN_SNOW'):
							self.iFeatureVariety = 2
						return
				

		if (self.lFeatureCount[3] >= self.lFeatureCount[0]) and \
			(self.lFeatureCount[3] >= self.lFeatureCount[1]) and \
			(self.lFeatureCount[3] >= self.lFeatureCount[2]) and \
			(self.lFeatureCount[3] >= self.lFeatureCount[4]) and \
			(self.lFeatureCount[3] >= self.lFeatureCount[5]) and \
			(self.lFeatureCount[3] >= self.lFeatureCount[6]):
				self.DebugLogging("\nPlot Feature set FEATURE_ICE")
##				if (self.iPlotY in range(2)) or (self.iPlotY in range(self.tDoneSize[1]-2, self.tDoneSize[1])):
##                                          self.iFeatureVariety = random.randint(1, 2)
##                                  elif (self.iPlotY in range(2, 5)) or (self.iPlotY in range(self.tDoneSize[1]-5, self.tDoneSize[1]-2)):
##                                          self.iFeatureVariety = random.randint(0, 2)
##                                  else:
##                                          self.iFeatureVariety = random.randint(0, 1)
				self.iFeatureVariety = 0
				self.sFeatureDesc = 'FEATURE_ICE'
				return

		# Oasis can only appear in desert
		if (self.sTerrainDesc == 'TERRAIN_DESERT'):
			if (self.lFeatureCount[4] >= self.lFeatureCount[0]) and \
				(self.lFeatureCount[4] >= self.lFeatureCount[1]) and \
				(self.lFeatureCount[4] >= self.lFeatureCount[2]) and \
				(self.lFeatureCount[4] >= self.lFeatureCount[3]) and \
				(self.lFeatureCount[4] >= self.lFeatureCount[5]) and \
				(self.lFeatureCount[4] >= self.lFeatureCount[6]):
					self.DebugLogging("\nPlot Feature set FEATURE_OASIS")
					self.iFeatureVariety = 0
					self.sFeatureDesc = 'FEATURE_OASIS'
					return

		# Flood Plains (if enabled) can only appear in desert
		if (self.sTerrainDesc == 'TERRAIN_DESERT'):
			if (self.lFeatureCount[5] >= self.lFeatureCount[0]) and \
				(self.lFeatureCount[5] >= self.lFeatureCount[1]) and \
				(self.lFeatureCount[5] >= self.lFeatureCount[2]) and \
				(self.lFeatureCount[5] >= self.lFeatureCount[3]) and \
				(self.lFeatureCount[5] >= self.lFeatureCount[4]) and \
				(self.lFeatureCount[5] >= self.lFeatureCount[6]) and \
                                ((self.iFeatureCoastsFlag == 1) or (self.iFeatureCoastsFlag == 3)):
					self.DebugLogging("\nPlot Feature set FEATURE_FLOOD_PLAINS")
					self.iFeatureVariety = 0
					self.sFeatureDesc = 'FEATURE_FLOOD_PLAINS'
					return

		# Fallout
		if (self.lFeatureCount[6] >= self.lFeatureCount[0]) and \
			(self.lFeatureCount[6] >= self.lFeatureCount[1]) and \
			(self.lFeatureCount[6] >= self.lFeatureCount[2]) and \
			(self.lFeatureCount[6] >= self.lFeatureCount[3]) and \
			(self.lFeatureCount[6] >= self.lFeatureCount[4]) and \
			(self.lFeatureCount[6] >= self.lFeatureCount[5]):
                                self.DebugLogging("\nPlot Feature set FEATURE_FALLOUT")
				self.iFeatureVariety = 0
				self.sFeatureDesc = 'FEATURE_FALLOUT'
				return

										
	def DetermineRiverData(self):		
		if not self.RiverMap:
			return
		self.DebugLogging("\nself.iRiverNtoSCount: %d\nself.iRiverStoNCount: %d\nself.iRiverEtoWCount: %d\nself.iRiverWtoECount: %d" %(self.iRiverNtoSCount, self.iRiverStoNCount, self.iRiverEtoWCount, self.iRiverWtoECount))
			
		# If there's not enough river pixels here then skip this stuff
		if (self.iRiverNtoSCount < self.iRiverSensitivity and self.iRiverStoNCount < self.iRiverSensitivity and \
			self.iRiverEtoWCount < self.iRiverSensitivity and self.iRiverWtoECount < self.iRiverSensitivity and \
			self.iRiverAngleWtoNCount < self.iRiverSensitivity and self.iRiverAngleNtoWCount < self.iRiverSensitivity and \
			self.iRiverAngleStoNWCount < self.iRiverSensitivity and self.iRiverAngleNWtoSCount < self.iRiverSensitivity):
				return
			
		if (self.iRiverNtoSCount >= self.iRiverStoNCount) and \
			(self.iRiverNtoSCount >= self.iRiverEtoWCount) and \
			(self.iRiverNtoSCount >= self.iRiverWtoECount) and \
			(self.iRiverNtoSCount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverNtoSCount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverNtoSCount >= self.iRiverAngleStoNWCount) and \
			(self.iRiverNtoSCount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists = "isWOfRiver"
				self.sRiverDirection = "RiverNSDirection=2"
				self.DebugLogging("\nPlot River set to isWofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverNSDirection=2")
					
		# River flows South to North
		if (self.iRiverStoNCount >= self.iRiverNtoSCount) and \
			(self.iRiverStoNCount >= self.iRiverEtoWCount) and \
			(self.iRiverStoNCount >= self.iRiverWtoECount) and \
			(self.iRiverStoNCount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverStoNCount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverStoNCount >= self.iRiverAngleStoNWCount) and \
			(self.iRiverStoNCount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists = "isWOfRiver"
				self.sRiverDirection = "RiverNSDirection=0"
				self.DebugLogging("\nPlot River set to isWofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverNSDirection=0")
					
		# River flows East to West
		if (self.iRiverEtoWCount >= self.iRiverNtoSCount) and \
			(self.iRiverEtoWCount >= self.iRiverStoNCount) and \
			(self.iRiverEtoWCount >= self.iRiverWtoECount) and \
			(self.iRiverEtoWCount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverEtoWCount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverEtoWCount >= self.iRiverAngleStoNWCount) and \
			(self.iRiverEtoWCount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists = "isNOfRiver"
				self.sRiverDirection = "RiverWEDirection=3"
				self.DebugLogging("\nPlot River set to isNofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverWEDirection=3")
					
		# River flows West to East
		if (self.iRiverWtoECount >= self.iRiverNtoSCount) and \
			(self.iRiverWtoECount >= self.iRiverStoNCount) and \
			(self.iRiverWtoECount >= self.iRiverEtoWCount) and \
			(self.iRiverWtoECount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverWtoECount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverWtoECount >= self.iRiverAngleStoNWCount) and \
			(self.iRiverWtoECount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists = "isNOfRiver"
				self.sRiverDirection = "RiverWEDirection=1"
				self.DebugLogging("\nPlot River set to isNofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverWEDirection=1")
					
		# River flows West to North
		if (self.iRiverAngleWtoNCount >= self.iRiverNtoSCount) and \
			(self.iRiverAngleWtoNCount >= self.iRiverStoNCount) and \
			(self.iRiverAngleWtoNCount >= self.iRiverEtoWCount) and \
			(self.iRiverAngleWtoNCount >= self.iRiverWtoECount) and \
			(self.iRiverAngleWtoNCount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverAngleWtoNCount >= self.iRiverAngleStoNWCount) and \
			(self.iRiverAngleWtoNCount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists = "isNOfRiver"
				self.sRiverDirection = "RiverWEDirection=1"
				self.sRiverExists2 = "isWOfRiver"
				self.sRiverDirection2 = "RiverNSDirection=0"
				self.DebugLogging("\nPlot River set to isNofRiver and isWofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverWEDirection=1 and RiverNSDirection=0")

		# River flows North to West
		if (self.iRiverAngleNtoWCount >= self.iRiverNtoSCount) and \
			(self.iRiverAngleNtoWCount >= self.iRiverStoNCount) and \
			(self.iRiverAngleNtoWCount >= self.iRiverEtoWCount) and \
			(self.iRiverAngleNtoWCount >= self.iRiverWtoECount) and \
			(self.iRiverAngleNtoWCount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverAngleNtoWCount >= self.iRiverAngleStoNWCount) and \
			(self.iRiverAngleNtoWCount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists2 = "isNOfRiver"
				self.sRiverDirection2 = "RiverWEDirection=3"
				self.sRiverExists = "isWOfRiver"
				self.sRiverDirection = "RiverNSDirection=2"
				self.DebugLogging("\nPlot River set to isNofRiver and isWofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverWEDirection=3 and RiverNSDirection=2")

		# River flows South to North and West
		if (self.iRiverAngleStoNWCount >= self.iRiverNtoSCount) and \
			(self.iRiverAngleStoNWCount >= self.iRiverStoNCount) and \
			(self.iRiverAngleStoNWCount >= self.iRiverEtoWCount) and \
			(self.iRiverAngleStoNWCount >= self.iRiverWtoECount) and \
			(self.iRiverAngleStoNWCount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverAngleStoNWCount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverAngleStoNWCount >= self.iRiverAngleNWtoSCount):
				self.sRiverExists = "isNOfRiver"
				self.sRiverDirection = "RiverWEDirection=0"
				self.sRiverExists2 = "isWOfRiver"
				self.sRiverDirection2 = "RiverNSDirection=0"
				self.DebugLogging("\nPlot River set to isNofRiver and isWofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverWEDirection=0 and RiverNSDirection=0")

		# River flows North and West to South
		if (self.iRiverAngleNWtoSCount >= self.iRiverNtoSCount) and \
			(self.iRiverAngleNWtoSCount >= self.iRiverStoNCount) and \
			(self.iRiverAngleNWtoSCount >= self.iRiverEtoWCount) and \
			(self.iRiverAngleNWtoSCount >= self.iRiverWtoECount) and \
			(self.iRiverAngleNWtoSCount >= self.iRiverAngleWtoNCount) and \
			(self.iRiverAngleNWtoSCount >= self.iRiverAngleNtoWCount) and \
			(self.iRiverAngleNWtoSCount >= self.iRiverAngleStoNWCount):
				self.sRiverExists = "isNOfRiver"
				self.sRiverDirection = "RiverWEDirection=1"
				self.sRiverExists2 = "isWOfRiver"
				self.sRiverDirection2 = "RiverNSDirection=2"
				self.DebugLogging("\nPlot River set to isNofRiver and isWofRiver")
				self.DebugLogging("\nPlot River Direction set to RiverWEDirection=1 and RiverNSDirection=2")

	def DetermineBonusData(self):
		if not self.BonusMap:
			return
		self.DebugLogging("\nDetermineBonusData")
		
		self.sBonusType = "NONE"
		self.iBonusFlag = 0
		if (self.lBonusCount[0] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[0] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_ALUMINUM"
					self.DebugLogging("\nPlot Bonus set to ALUMINUM")
					self.lBonusStats[0] += 1
					return
		if (self.lBonusCount[1] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[1] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_COAL"
					self.DebugLogging("\nPlot Bonus set to COAL")
					self.lBonusStats[1] += 1					
					return
		if (self.lBonusCount[2] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[2] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_COPPER"
					self.DebugLogging("\nPlot Bonus set to COPPER")
					self.lBonusStats[2] += 1
					return
		if (self.lBonusCount[3] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[3] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_HORSE"
					self.DebugLogging("\nPlot Bonus set to HORSE")
					self.lBonusStats[3] += 1
					return
		if (self.lBonusCount[4] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[4] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_IRON"
					self.DebugLogging("\nPlot Bonus set to IRON")
					self.lBonusStats[4] += 1
					return                                
		if (self.lBonusCount[5] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[5] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_MARBLE"
					self.DebugLogging("\nPlot Bonus set to MARBLE")
					self.lBonusStats[5] += 1
					return
		if (self.lBonusCount[6] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[6] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 4):
					self.sBonusType = "BONUS_OIL"
					self.DebugLogging("\nPlot Bonus set to OIL")
					self.lBonusStats[6] += 1
					return 
		if (self.lBonusCount[7] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[7] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_STONE"
					self.DebugLogging("\nPlot Bonus set to STONE")
					self.lBonusStats[7] += 1
					return
		if (self.lBonusCount[8] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[8] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_URANIUM"
					self.DebugLogging("\nPlot Bonus set to URANIUM")
					self.lBonusStats[8] += 1
					return
		if (self.lBonusCount[9] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[9] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_DYE"
					self.DebugLogging("\nPlot Bonus set to DYE")
					self.lBonusStats[9] += 1
					return
		if (self.lBonusCount[10] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[10] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_FUR"
					self.DebugLogging("\nPlot Bonus set to FUR")
					self.lBonusStats[10] += 1
					return
		if (self.lBonusCount[11] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[11] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_GEMS"
					self.DebugLogging("\nPlot Bonus set to GEMS")
					self.lBonusStats[11] += 1
					return
		if (self.lBonusCount[12] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[12] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_GOLD"
					self.DebugLogging("\nPlot Bonus set to GOLD")
					self.lBonusStats[12] += 1
					return
		if (self.lBonusCount[13] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[13] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_INCENSE"
					self.DebugLogging("\nPlot Bonus set to INCENSE")
					self.lBonusStats[13] += 1
					return
		if (self.lBonusCount[14] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[14] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_IVORY"
					self.DebugLogging("\nPlot Bonus set to IVORY")
					self.lBonusStats[14] += 1
					return
		if (self.lBonusCount[15] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[15] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_SILK"
					self.DebugLogging("\nPlot Bonus set to SILK")
					self.lBonusStats[15] += 1
					return
		if (self.lBonusCount[16] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[16] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_SILVER"
					self.DebugLogging("\nPlot Bonus set to SILVER")
					self.lBonusStats[16] += 1
					return
		if (self.lBonusCount[17] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[17] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_SPICES"
					self.DebugLogging("\nPlot Bonus set to SPICES")
					self.lBonusStats[17] += 1
					return
		if (self.lBonusCount[18] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[18] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_SUGAR"
					self.DebugLogging("\nPlot Bonus set to SUGAR")
					self.lBonusStats[18] += 1
					return
		if (self.lBonusCount[19] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[19] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_WINE"
					self.DebugLogging("\nPlot Bonus set to WINE")
					self.lBonusStats[19] += 1
					return
		if (self.lBonusCount[20] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[20] < self.lBonusCount[i]):
					break
				elif (self.iPlotType > 2):
					self.sBonusType = "BONUS_WHALE"
					self.DebugLogging("\nPlot Bonus set to WHALE")
					self.lBonusStats[20] += 1
					return
		if (self.lBonusCount[21] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[21] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_BANANA"
					self.DebugLogging("\nPlot Bonus set to BANANA")
					self.lBonusStats[21] += 1
					return
		if (self.lBonusCount[22] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[22] < self.lBonusCount[i]):
					break
				elif (self.iPlotType > 2):
					self.sBonusType = "BONUS_CLAM"
					self.DebugLogging("\nPlot Bonus set to CLAM")
					self.lBonusStats[22] += 1
					return
		if (self.lBonusCount[23] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[23] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_CORN"
					self.DebugLogging("\nPlot Bonus set to CORN")
					self.lBonusStats[23] += 1
					return
		if (self.lBonusCount[24] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[24] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_COW"
					self.DebugLogging("\nPlot Bonus set to COW")
					self.lBonusStats[24] += 1
					return
		if (self.lBonusCount[25] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[25] < self.lBonusCount[i]):
					break
				elif (self.iPlotType > 2):
					self.sBonusType = "BONUS_CRAB"
					self.DebugLogging("\nPlot Bonus set to CRAB")
					self.lBonusStats[25] += 1
					return
		if (self.lBonusCount[26] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[26] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_DEER"
					self.DebugLogging("\nPlot Bonus set to DEER")
					self.lBonusStats[26] += 1
					return
		if (self.lBonusCount[27] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[27] < self.lBonusCount[i]):
					break
				elif (self.iPlotType > 2):
					self.sBonusType = "BONUS_FISH"
					self.DebugLogging("\nPlot Bonus set to FISH")
					self.lBonusStats[27] += 1
					return
		if (self.lBonusCount[28] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[28] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_PIG"
					self.DebugLogging("\nPlot Bonus set to PIG")
					self.lBonusStats[28] += 1
					return
		if (self.lBonusCount[29] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[29] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_RICE"
					self.DebugLogging("\nPlot Bonus set to RICE")
					self.lBonusStats[29] += 1
					return
		if (self.lBonusCount[30] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[30] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_SHEEP"
					self.DebugLogging("\nPlot Bonus set to SHEEP")
					self.lBonusStats[30] += 1
					return
		if (self.lBonusCount[31] != 0):
			for i in range( self.iNumBonusColorPalettes ):
				if (self.lBonusCount[31] < self.lBonusCount[i]):
					break
				elif (self.iPlotType < 3):
					self.sBonusType = "BONUS_WHEAT"
					self.DebugLogging("\nPlot Bonus set to WHEAT")
					self.lBonusStats[31] += 1
					return

	def AddCoastlines(self, iPlotX, iPlotY):
		self.DebugLogging("\nDetermineCoastlines")
		# Turn Ocean to Coast where necessary	
		# If this is ocean...
		if (self.aaiPlotTypes[iPlotX][iPlotY] == 3):
			# Check adjacent plots
			for iXOffset in range(-1,2):
				for iYOffset in range(-1,2):
					# Check to see if its a plot that actually exists
					if ((iPlotX + iXOffset > 0) and (iPlotX + iXOffset < self.iSizeX) and (iPlotY + iYOffset > 0) and (iPlotY + iYOffset < self.iSizeY)):
						# If adjacent plot is land (types 0, 1 or 2)
						if (self.aaiPlotTypes[iPlotX + iXOffset][iPlotY + iYOffset] < 3):
							# Turn active ocean plot to coast
							self.aasTerrainTypes[iPlotX][iPlotY] = "TERRAIN_COAST"
							self.DebugLogging("\nTurning ocean plot to coast")

	def AddFloodPlains(self, iPlotX, iPlotY):
		self.DebugLogging("\nDetermineFloodPlains")
		# Add Flood Plains where necessary			
		# If this is desert...
		if (self.aasTerrainTypes[iPlotX][iPlotY] == 'TERRAIN_DESERT'):
			# Check adjacent plots
			for iXOffset in range(-1,1):
				for iYOffset in range(0,2):
					# Check to see if its a plot that actually exists
					if ((iPlotX + iXOffset > 0) and (iPlotX + iXOffset < self.iSizeX) and (iPlotY + iYOffset > 0) and (iPlotY + iYOffset < self.iSizeY)):
						# If adjacent plot has a river
						if (self.aasRiverExists[iPlotX + iXOffset][iPlotY + iYOffset]):
							# Add Flood Plains
							self.aasFeatureTypes[iPlotX][iPlotY] = "FEATURE_FLOOD_PLAINS"
							self.aaiFeatureVTypes[iPlotX][iPlotY] = random.randint(0, 1)
							self.DebugLogging("\nAdding flood plain")


	def WriteImprovements(self, iPlotX, iPlotY):
                for tPlotCoords in scenario.dltFBoats:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_FISHING_BOATS\n")

                for tPlotCoords in scenario.dltWBoats:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_WHALING_BOATS\n")

                for tPlotCoords in scenario.dltMine:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_MINE\n")

                for tPlotCoords in scenario.dltQuarry:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_QUARRY\n")

                for tPlotCoords in scenario.dltCamp:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_CAMP\n")

                for tPlotCoords in scenario.dltFarm:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_FARM\n")

                for tPlotCoords in scenario.dltPlantation:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_PLANTATION\n")

                for tPlotCoords in scenario.dltWinery:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_WINERY\n")

                for tPlotCoords in scenario.dltPasture:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_PASTURE\n")

                for tPlotCoords in scenario.dltWindmill:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_WINDMILL\n")

                for tPlotCoords in scenario.dltLumbermill:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_LUMBERMILL\n")

                for tPlotCoords in scenario.dltWatermill:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_WATERMILL\n")

                for tPlotCoords in scenario.dltFort:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_FORT\n")

                for tPlotCoords in scenario.dltOPlatform:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_OFFSHORE_PLATFORM\n")

                for tPlotCoords in scenario.dltWell:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_WELL\n")

                for tPlotCoords in scenario.dltWorkshop:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_WORKSHOP\n")

                for tPlotCoords in scenario.dltCottage:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_COTTAGE\n")

                for tPlotCoords in scenario.dltHamlet:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_HAMLET\n")
                                
                for tPlotCoords in scenario.dltVillage:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_VILLAGE\n")

                for tPlotCoords in scenario.dltTown:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_TOWN\n")

                for tPlotCoords in scenario.dltRuins:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_CITY_RUINS\n")

                for tPlotCoords in scenario.dltGoodyHut:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tImprovementType=IMPROVEMENT_GOODY_HUT\n")

                for tPlotCoords in scenario.dltRoad:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tRouteType=ROUTE_ROAD\n")

                for tPlotCoords in scenario.dltRailroad:
                        if (tPlotCoords == (iPlotX, iPlotY)):
                                self.OutputFile.write("\tRouteType=ROUTE_RAILROAD\n")


	def WriteFOW(self, iPlotX, iPlotY):
 		for iPlayer in range(scenario.iDefaultNumberOfCivs):
                        for tPlotCoords in scenario.dltRevealPlots[iPlayer]:
                                if (tPlotCoords == (iPlotX, iPlotY)):
                                        self.OutputFile.write("\tTeamReveal=")
                        		for iPlayer in range(scenario.iDefaultNumberOfCivs):
                                                for tPlotCoords in scenario.dltRevealPlots[iPlayer]:
                                                        if (tPlotCoords == (iPlotX, iPlotY)):
                                                                #self.DebugLogging("X=%d ; Y=%d\n" % (tPlotCoords[0], tPlotCoords[1]))                                        
                                                                self.OutputFile.write("%d," % iPlayer)
                                                                self.DebugLogging("\tTeam%dPlotRevealed=1" % iPlayer)
                                        self.OutputFile.write("\n")
                                        return


		
	def OutputToFile(self, iPlotX, iPlotY):
		global scenario
		self.aasTerrainTypes[self.iPlotX][self.iPlotY] = self.sTerrainDesc
		self.aasFeatureTypes[self.iPlotX][self.iPlotY] = self.sFeatureDesc
		self.aaiFeatureVTypes[self.iPlotX][self.iPlotY] = self.iFeatureVariety
		self.aaiPlotTypes[self.iPlotX][self.iPlotY] = self.iPlotType


		# Terrain changes for the scenario
                scenario.TerrainChanges(self, iPlotX, iPlotY)
                                

		self.DebugLogging("\nOutputToFile: plot (%d, %d)" %(iPlotX, iPlotY))
		self.OutputFile.write("BeginPlot\n")
		self.OutputFile.write("\tx=%d,y=%d\n" %(iPlotX, iPlotY))
		# Write out bonus if it exists
		if (self.aasBonusType[iPlotX][iPlotY] != "NONE"):
			self.OutputFile.write("\tBonusType=%s\n" %(self.aasBonusType[iPlotX][iPlotY]))
		# Write out river if it exists
		if (self.aasRiverExists[iPlotX][iPlotY]):
			self.OutputFile.write("\t%s\n" %(self.aasRiverExists[iPlotX][iPlotY]))
			self.OutputFile.write("\t%s\n" %(self.aasRiverDirection[iPlotX][iPlotY]))
		if (self.aasRiverExists2[iPlotX][iPlotY]):
			self.OutputFile.write("\t%s\n" %(self.aasRiverExists2[iPlotX][iPlotY]))
			self.OutputFile.write("\t%s\n" %(self.aasRiverDirection2[iPlotX][iPlotY]))
		# Write feature data for this plot only if it actually has a feature
		if self.aasFeatureTypes[iPlotX][iPlotY] != "NONE" and self.aaiFeatureVTypes[iPlotX][iPlotY] != -1:
			self.OutputFile.write("\tFeatureType=%s, FeatureVariety=%d\n" %(self.aasFeatureTypes[iPlotX][iPlotY], self.aaiFeatureVTypes[iPlotX][iPlotY]))
		self.OutputFile.write("\tTerrainType=%s\n" %(self.aasTerrainTypes[iPlotX][iPlotY],))
		self.OutputFile.write("\tPlotType=%d\n" %(self.aaiPlotTypes[iPlotX][iPlotY]))

		# Write terrain improvements
                self.WriteImprovements(iPlotX, iPlotY)

		# Write starting plots
                scenario.WriteStartingPlots(self, iPlotX, iPlotY)

		# Write cities and units
                scenario.WriteCitiesUnits(self, iPlotX, iPlotY)

		# Write fog of war                        
                self.WriteFOW(iPlotX, iPlotY)

		self.OutputFile.write("EndPlot\n")

		
	def DebugText(self, iXStop, iYStop):
		if self.iPlotX == iXStop and self.iPlotY == iYStop:
			print("Plot XY    : %d, %d" %(self.iPlotX, self.iPlotY))
			print("Colors     : %d, %d, %d" %(self.tTerrainPixel[0], self.tTerrainPixel[1], self.tTerrainPixel[2]))
			print("Pixel Loop : %d, %d" %(self.iPixelX, self.iPixelY))
			print("Real Pixel : %d, %d" %(self.iActivePixelX, self.iActivePixelY))
			print("Process X  : (%d * %d) + %d + %d" %(self.iPlotX, self.fFactor[0], self.iPixelX, self.iXOffset))
			print("Process Y  : %d - ((%d * %d) + %d + %d)" %(self.BMPSize[1], self.iPlotY, self.fFactor[1], self.iPixelY, self.iYOffset))
			print("\n")
	

	def getExtraWBSText_WorldSize(self):
                WBSText = "\n\nBeginMap\n\tgrid width=%d\n\tgrid height=%d" %(self.iSizeX, self.iSizeY)
		return WBSText

	def getExtraWBSText_Wrap(self):
                if (self.iWrapX == 1):
                        if (self.iWrapY == 1):
                                WBSText = "\n\twrap X=1\n\twrap Y=1"
                                return WBSText
                        else:
                                WBSText = "\n\twrap X=1\n\twrap Y=0"
                                return WBSText
                else:
                        if (self.iWrapY == 1):
                                WBSText = "\n\twrap X=0\n\twrap Y=1"
                                return WBSText
                        else:
                                WBSText = "\n\twrap X=0\n\twrap Y=0"
                                return WBSText

	def FOWExtract(self):
		# Set up images to be loaded
                if not self.FOWMap:
                        print("Cancelled!")
                        return 1
		self.FOWMap = Image.open(self.FOWMap)
				
		self.BMPSize = self.FOWMap.size
		
		if Debug:
			self.DebugLog = open("DebugLog.txt", 'w')
			#self.DebugLogging("\n__init__\n\n")

		self.FOW_Output = open("FOW_Output.txt", 'w')

		print("Processing Image...") 

		# self.tDoneSize is the size of the map in CIV
		self.tDoneSize = (self.BMPSize[0],self.BMPSize[1])
		
		self.iActivePixelX = 0
		self.iActivePixelY = 0

		self.iCounter = 0

		# For the FoW count
		self.iActivePixelX_old = 0

		self.ListPixels()	
		
		if Debug:
			self.DebugLog.close()

		self.FOW_Output.close()				
						
	def ListPixels(self):
		if not self.FOWMap:
			return
		    
		for self.iActivePixelX in range(self.tDoneSize[0]):
			for self.iActivePixelY in range(self.tDoneSize[1] -1, 0, -1):
                            	#self.DebugLogging("\nX: %d \t Y: %d" %(self.iActivePixelX, self.iActivePixelY))
                                self.tPixel = self.FOWMap.getpixel((self.iActivePixelX, self.iActivePixelY))
                                if (self.tPixel == d_ColorPalettes.get('FOW_LAND')) or (self.tPixel == d_ColorPalettes.get('FOW_WATER')):
                                        self.iCounter += 1
                                        if (self.iCounter == 1):
                                                self.FOW_Output.write("\n                                        (%d, %d)" %(self.iActivePixelX, self.tDoneSize[1] - self.iActivePixelY - 1))
                                        elif (self.iActivePixelX != self.iActivePixelX_old):
                                                self.FOW_Output.write(",\n                                        (%d, %d)" %(self.iActivePixelX, self.tDoneSize[1] - self.iActivePixelY - 1))
                                        else:
                                                self.FOW_Output.write(", (%d, %d)" %(self.iActivePixelX, self.tDoneSize[1] - self.iActivePixelY - 1))
                                        #remembers the last X
                                        self.iActivePixelX_old = self.iActivePixelX

                print("\nNumber of plots: %d" %(self.iCounter))


	
	def DebugLogging(self, msg):
		if Debug:
			self.DebugLog.write(msg)

#runBMPConvertToWBSApp()
